/*
 * Decompiled with CFR 0.152.
 */
package net.runelite.client.plugins.gpu;

import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.ShortBuffer;
import javax.inject.Singleton;
import net.runelite.api.Scene;
import net.runelite.client.plugins.gpu.GLBuffer;
import net.runelite.client.plugins.gpu.template.Template;
import net.runelite.client.util.OSType;
import net.runelite.rlawt.AWTContext;
import org.lwjgl.PointerBuffer;
import org.lwjgl.opencl.APPLEGLSharing;
import org.lwjgl.opencl.CL;
import org.lwjgl.opencl.CL10;
import org.lwjgl.opencl.CL10GL;
import org.lwjgl.opencl.CL11;
import org.lwjgl.opencl.CL12;
import org.lwjgl.opencl.CLCapabilities;
import org.lwjgl.opencl.CLContextCallback;
import org.lwjgl.opencl.CLContextCallbackI;
import org.lwjgl.opencl.CLImageFormat;
import org.lwjgl.system.Configuration;
import org.lwjgl.system.MemoryStack;
import org.lwjgl.system.MemoryUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
class OpenCLManager {
    private static final Logger log = LoggerFactory.getLogger(OpenCLManager.class);
    private static final String KERNEL_NAME_UNORDERED = "computeUnordered";
    private static final String KERNEL_NAME_LARGE = "computeLarge";
    private static final int MIN_WORK_GROUP_SIZE = 256;
    private static final int SMALL_SIZE = 512;
    private static final int LARGE_SIZE = 6144;
    private static final int SHARED_SIZE = 43;
    private boolean initialized;
    private int largeFaceCount;
    private int smallFaceCount;
    private long device;
    long context;
    private long commandQueue;
    private long programUnordered;
    private long programSmall;
    private long programLarge;
    private long kernelUnordered;
    private long kernelSmall;
    private long kernelLarge;
    private long tileHeightImage;

    OpenCLManager() {
    }

    void init(AWTContext awtContext) {
        this.commandQueue = 0L;
        this.context = 0L;
        this.device = 0L;
        this.programLarge = 0L;
        this.programSmall = 0L;
        this.programUnordered = 0L;
        this.kernelLarge = 0L;
        this.kernelSmall = 0L;
        this.kernelUnordered = 0L;
        this.tileHeightImage = 0L;
        CL.create();
        this.initialized = true;
        try (MemoryStack stack = MemoryStack.stackPush();){
            if (OSType.getOSType() == OSType.MacOS) {
                this.initContextMacOS(awtContext, stack);
            } else {
                this.initContext(awtContext, stack);
            }
            this.ensureMinWorkGroupSize();
            this.initQueue();
            this.compilePrograms(stack);
        }
    }

    void cleanup() {
        if (!this.initialized) {
            return;
        }
        try {
            if (this.tileHeightImage != 0L) {
                CL12.clReleaseMemObject(this.tileHeightImage);
            }
            CL12.clReleaseKernel(this.kernelUnordered);
            CL12.clReleaseKernel(this.kernelSmall);
            CL12.clReleaseKernel(this.kernelLarge);
            CL12.clReleaseProgram(this.programUnordered);
            CL12.clReleaseProgram(this.programSmall);
            CL12.clReleaseProgram(this.programLarge);
            CL12.clReleaseCommandQueue(this.commandQueue);
            CL12.clReleaseContext(this.context);
            CL12.clReleaseDevice(this.device);
        }
        finally {
            CL.destroy();
            this.initialized = false;
        }
    }

    private void initContext(AWTContext awtContext, MemoryStack stack) {
        IntBuffer pi = stack.mallocInt(1);
        OpenCLManager.checkCLError(CL11.clGetPlatformIDs(null, pi));
        if (pi.get(0) == 0) {
            throw new RuntimeException("No OpenCL platforms found.");
        }
        PointerBuffer platforms = stack.mallocPointer(pi.get(0));
        OpenCLManager.checkCLError(CL11.clGetPlatformIDs(platforms, (IntBuffer)null));
        PointerBuffer ctxProps = stack.mallocPointer(7);
        if (OSType.getOSType() == OSType.Windows) {
            ctxProps.put(4228L).put(0L).put(8200L).put(awtContext.getGLContext()).put(8203L).put(awtContext.getWGLHDC()).put(0L).flip();
        } else if (OSType.getOSType() == OSType.Linux) {
            ctxProps.put(4228L).put(0L).put(8200L).put(awtContext.getGLContext()).put(8202L).put(awtContext.getGLXDisplay()).put(0L).flip();
        } else {
            throw new RuntimeException("unsupported platform");
        }
        for (int p = 0; p < platforms.capacity(); ++p) {
            long platform = platforms.get(p);
            ctxProps.put(1, platform);
            try {
                CLCapabilities platformCaps = CL.createPlatformCapabilities(platform);
                log.debug("Platform profile: {}", (Object)OpenCLManager.getPlatformInfoStringUTF8(platform, 2304));
                log.debug("Platform version: {}", (Object)OpenCLManager.getPlatformInfoStringUTF8(platform, 2305));
                log.debug("Platform name: {}", (Object)OpenCLManager.getPlatformInfoStringUTF8(platform, 2306));
                log.debug("Platform vendor: {}", (Object)OpenCLManager.getPlatformInfoStringUTF8(platform, 2307));
                log.debug("Platform extensions: {}", (Object)OpenCLManager.getPlatformInfoStringUTF8(platform, 2308));
                OpenCLManager.checkCLError(CL11.clGetDeviceIDs(platform, 4L, null, pi));
                PointerBuffer devices = stack.mallocPointer(pi.get(0));
                OpenCLManager.checkCLError(CL11.clGetDeviceIDs(platform, 4L, devices, (IntBuffer)null));
                for (int d = 0; d < devices.capacity(); ++d) {
                    long device = devices.get(d);
                    try {
                        CLCapabilities deviceCaps = CL.createDeviceCapabilities(device, platformCaps);
                        log.debug("Device id {}", (Object)device);
                        log.debug("\tCL_DEVICE_NAME: {}", (Object)OpenCLManager.getDeviceInfoStringUTF8(device, 4139));
                        log.debug("\tCL_DEVICE_VENDOR: {}", (Object)OpenCLManager.getDeviceInfoStringUTF8(device, 4140));
                        log.debug("\tCL_DRIVER_VERSION: {}", (Object)OpenCLManager.getDeviceInfoStringUTF8(device, 4141));
                        log.debug("\tCL_DEVICE_PROFILE: {}", (Object)OpenCLManager.getDeviceInfoStringUTF8(device, 4142));
                        log.debug("\tCL_DEVICE_VERSION: {}", (Object)OpenCLManager.getDeviceInfoStringUTF8(device, 4143));
                        log.debug("\tCL_DEVICE_EXTENSIONS: {}", (Object)OpenCLManager.getDeviceInfoStringUTF8(device, 4144));
                        log.debug("\tCL_DEVICE_TYPE: {}", (Object)OpenCLManager.getDeviceInfoLong(device, 4096));
                        log.debug("\tCL_DEVICE_VENDOR_ID: {}", (Object)OpenCLManager.getDeviceInfoInt(device, 4097));
                        log.debug("\tCL_DEVICE_MAX_COMPUTE_UNITS: {}", (Object)OpenCLManager.getDeviceInfoInt(device, 4098));
                        log.debug("\tCL_DEVICE_MAX_WORK_ITEM_DIMENSIONS: {}", (Object)OpenCLManager.getDeviceInfoInt(device, 4099));
                        log.debug("\tCL_DEVICE_MAX_WORK_GROUP_SIZE: {}", (Object)OpenCLManager.getDeviceInfoPointer(device, 4100));
                        log.debug("\tCL_DEVICE_MAX_CLOCK_FREQUENCY: {}", (Object)OpenCLManager.getDeviceInfoInt(device, 4108));
                        log.debug("\tCL_DEVICE_ADDRESS_BITS: {}", (Object)OpenCLManager.getDeviceInfoInt(device, 4109));
                        log.debug("\tCL_DEVICE_AVAILABLE: {}", (Object)(OpenCLManager.getDeviceInfoInt(device, 4135) != 0 ? 1 : 0));
                        log.debug("\tCL_DEVICE_COMPILER_AVAILABLE: {}", (Object)(OpenCLManager.getDeviceInfoInt(device, 4136) != 0 ? 1 : 0));
                        if (!deviceCaps.cl_khr_gl_sharing && !deviceCaps.cl_APPLE_gl_sharing) continue;
                        IntBuffer errcode_ret = stack.callocInt(1);
                        long context = CL11.clCreateContext(ctxProps, device, (CLContextCallbackI)CLContextCallback.create((errinfo, private_info, cb, user_data) -> log.error("[LWJGL] cl_context_callback: {}", (Object)MemoryUtil.memUTF8(errinfo))), 0L, errcode_ret);
                        OpenCLManager.checkCLError(errcode_ret);
                        this.device = device;
                        this.context = context;
                        return;
                    }
                    catch (Exception ex) {
                        log.error("error checking device", ex);
                    }
                }
                continue;
            }
            catch (Exception ex) {
                log.error("error checking platform", ex);
            }
        }
        throw new RuntimeException("Unable to find compute platform");
    }

    private void initContextMacOS(AWTContext awtContext, MemoryStack stack) {
        PointerBuffer ctxProps = stack.mallocPointer(3);
        ctxProps.put(0x10000000L).put(awtContext.getCGLShareGroup()).put(0L).flip();
        IntBuffer errcode_ret = stack.callocInt(1);
        PointerBuffer devices = stack.mallocPointer(0);
        long context = CL11.clCreateContext(ctxProps, devices, (CLContextCallbackI)CLContextCallback.create((errinfo, private_info, cb, user_data) -> log.error("[LWJGL] cl_context_callback: {}", (Object)MemoryUtil.memUTF8(errinfo))), 0L, errcode_ret);
        OpenCLManager.checkCLError(errcode_ret);
        PointerBuffer deviceBuf = stack.mallocPointer(1);
        OpenCLManager.checkCLError(APPLEGLSharing.clGetGLContextInfoAPPLE(context, awtContext.getGLContext(), 0x10000002, deviceBuf, null));
        long device = deviceBuf.get(0);
        log.debug("Got macOS CLGL compute device {}", (Object)device);
        this.context = context;
        this.device = device;
    }

    private void ensureMinWorkGroupSize() {
        long[] maxWorkGroupSize = new long[1];
        CL12.clGetDeviceInfo(this.device, 4100, maxWorkGroupSize, null);
        log.debug("Device CL_DEVICE_MAX_WORK_GROUP_SIZE: {}", (Object)maxWorkGroupSize[0]);
        if (maxWorkGroupSize[0] < 256L) {
            throw new RuntimeException("Compute device does not support min work group size 256");
        }
        int groupSize = Integer.MIN_VALUE >>> Integer.numberOfLeadingZeros((int)maxWorkGroupSize[0]);
        this.largeFaceCount = 6144 / Math.min(groupSize, 6144);
        this.smallFaceCount = 512 / Math.min(groupSize, 512);
        log.debug("Face counts: small: {}, large: {}", (Object)this.smallFaceCount, (Object)this.largeFaceCount);
    }

    private void initQueue() {
        long[] l = new long[1];
        CL12.clGetDeviceInfo(this.device, 4138, l, null);
        this.commandQueue = CL12.clCreateCommandQueue(this.context, this.device, l[0] & 1L, (int[])null);
        log.debug("Created command_queue {}, properties {}", (Object)this.commandQueue, (Object)(l[0] & 1L));
    }

    private long compileProgram(MemoryStack stack, String programSource) {
        log.trace("Compiling program:\n {}", (Object)programSource);
        IntBuffer errcode_ret = stack.callocInt(1);
        long program = CL12.clCreateProgramWithSource(this.context, programSource, errcode_ret);
        OpenCLManager.checkCLError(errcode_ret);
        int err = CL12.clBuildProgram(program, this.device, (CharSequence)"", null, 0L);
        if (err != 0) {
            String errstr = OpenCLManager.getProgramBuildInfoStringASCII(program, this.device, 4483);
            throw new RuntimeException(errstr);
        }
        log.debug("Build status: {}", (Object)OpenCLManager.getProgramBuildInfoInt(program, this.device, 4481));
        log.debug("Binary type: {}", (Object)OpenCLManager.getProgramBuildInfoInt(program, this.device, 4484));
        log.debug("Build options: {}", (Object)OpenCLManager.getProgramBuildInfoStringASCII(program, this.device, 4482));
        log.debug("Build log: {}", (Object)OpenCLManager.getProgramBuildInfoStringASCII(program, this.device, 4483));
        return program;
    }

    private long getKernel(MemoryStack stack, long program, String kernelName) {
        IntBuffer errcode_ret = stack.callocInt(1);
        long kernel = CL12.clCreateKernel(program, (CharSequence)kernelName, errcode_ret);
        OpenCLManager.checkCLError(errcode_ret);
        log.debug("Loaded kernel {} for program {}", (Object)kernelName, (Object)program);
        return kernel;
    }

    private void compilePrograms(MemoryStack stack) {
        Template templateSmall = new Template().addInclude(OpenCLManager.class).add(key -> key.equals("FACE_COUNT") ? "#define FACE_COUNT " + this.smallFaceCount : null);
        Template templateLarge = new Template().addInclude(OpenCLManager.class).add(key -> key.equals("FACE_COUNT") ? "#define FACE_COUNT " + this.largeFaceCount : null);
        String unordered = new Template().addInclude(OpenCLManager.class).load("comp_unordered.cl");
        String small = templateSmall.load("comp.cl");
        String large = templateLarge.load("comp.cl");
        this.programUnordered = this.compileProgram(stack, unordered);
        this.programSmall = this.compileProgram(stack, small);
        this.programLarge = this.compileProgram(stack, large);
        this.kernelUnordered = this.getKernel(stack, this.programUnordered, KERNEL_NAME_UNORDERED);
        this.kernelSmall = this.getKernel(stack, this.programSmall, KERNEL_NAME_LARGE);
        this.kernelLarge = this.getKernel(stack, this.programLarge, KERNEL_NAME_LARGE);
    }

    void uploadTileHeights(Scene scene) {
        if (this.tileHeightImage != 0L) {
            CL12.clReleaseMemObject(this.tileHeightImage);
            this.tileHeightImage = 0L;
        }
        int TILEHEIGHT_BUFFER_SIZE = 270848;
        ShortBuffer tileBuffer = MemoryUtil.memAllocShort(270848);
        int[][][] tileHeights = scene.getTileHeights();
        for (int z = 0; z < 4; ++z) {
            for (int y = 0; y < 184; ++y) {
                for (int x = 0; x < 184; ++x) {
                    int h = tileHeights[z][x][y];
                    assert ((h & 7) == 0);
                    tileBuffer.put((short)(h >>= 3));
                }
            }
        }
        tileBuffer.flip();
        try (MemoryStack stack = MemoryStack.stackPush();){
            CLImageFormat imageFormat = CLImageFormat.calloc(stack);
            imageFormat.image_channel_order(4272);
            imageFormat.image_channel_data_type(4312);
            IntBuffer errcode_ret = stack.callocInt(1);
            this.tileHeightImage = CL12.clCreateImage3D(this.context, 36L, imageFormat, 184L, 184L, 4L, 0L, 0L, tileBuffer, errcode_ret);
            OpenCLManager.checkCLError(errcode_ret);
        }
        MemoryUtil.memFree(tileBuffer);
    }

    void compute(int unorderedModels, int smallModels, int largeModels, GLBuffer sceneVertexBuffer, GLBuffer sceneUvBuffer, GLBuffer vertexBuffer, GLBuffer uvBuffer, GLBuffer unorderedBuffer, GLBuffer smallBuffer, GLBuffer largeBuffer, GLBuffer outVertexBuffer, GLBuffer outUvBuffer, GLBuffer uniformBuffer) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            PointerBuffer glBuffers = stack.mallocPointer(10);
            glBuffers.put(sceneVertexBuffer.clBuffer);
            glBuffers.put(sceneUvBuffer.clBuffer);
            glBuffers.put(unorderedBuffer.clBuffer);
            glBuffers.put(smallBuffer.clBuffer);
            glBuffers.put(largeBuffer.clBuffer);
            glBuffers.put(vertexBuffer.clBuffer);
            glBuffers.put(uvBuffer.clBuffer);
            glBuffers.put(outVertexBuffer.clBuffer);
            glBuffers.put(outUvBuffer.clBuffer);
            glBuffers.put(uniformBuffer.clBuffer);
            glBuffers.flip();
            PointerBuffer acquireEvent = stack.mallocPointer(1);
            CL10GL.clEnqueueAcquireGLObjects(this.commandQueue, glBuffers, null, acquireEvent);
            PointerBuffer computeEvents = stack.mallocPointer(3);
            if (unorderedModels > 0) {
                CL12.clSetKernelArg1p(this.kernelUnordered, 0, unorderedBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelUnordered, 1, sceneVertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelUnordered, 2, vertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelUnordered, 3, sceneUvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelUnordered, 4, uvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelUnordered, 5, outVertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelUnordered, 6, outUvBuffer.clBuffer);
                CL12.clEnqueueNDRangeKernel(this.commandQueue, this.kernelUnordered, 1, null, stack.pointers((long)unorderedModels * 6L), stack.pointers(6L), acquireEvent, computeEvents);
                computeEvents.position(computeEvents.position() + 1);
            }
            if (smallModels > 0) {
                CL12.clSetKernelArg(this.kernelSmall, 0, 2220L);
                CL12.clSetKernelArg1p(this.kernelSmall, 1, smallBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 2, sceneVertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 3, vertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 4, sceneUvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 5, uvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 6, outVertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 7, outUvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelSmall, 8, uniformBuffer.clBuffer);
                CL12.clSetKernelArg1l(this.kernelSmall, 9, this.tileHeightImage);
                CL12.clEnqueueNDRangeKernel(this.commandQueue, this.kernelSmall, 1, null, stack.pointers((long)(smallModels * (512 / this.smallFaceCount))), stack.pointers((long)(512 / this.smallFaceCount)), acquireEvent, computeEvents);
                computeEvents.position(computeEvents.position() + 1);
            }
            if (largeModels > 0) {
                CL12.clSetKernelArg(this.kernelLarge, 0, 24748L);
                CL12.clSetKernelArg1p(this.kernelLarge, 1, largeBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 2, sceneVertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 3, vertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 4, sceneUvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 5, uvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 6, outVertexBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 7, outUvBuffer.clBuffer);
                CL12.clSetKernelArg1p(this.kernelLarge, 8, uniformBuffer.clBuffer);
                CL12.clSetKernelArg1l(this.kernelLarge, 9, this.tileHeightImage);
                CL12.clEnqueueNDRangeKernel(this.commandQueue, this.kernelLarge, 1, null, stack.pointers((long)(largeModels * (6144 / this.largeFaceCount))), stack.pointers((long)(6144 / this.largeFaceCount)), acquireEvent, computeEvents);
                computeEvents.position(computeEvents.position() + 1);
            }
            if (computeEvents.position() == 0) {
                CL10GL.clEnqueueReleaseGLObjects(this.commandQueue, glBuffers, null, null);
            } else {
                computeEvents.flip();
                CL10GL.clEnqueueReleaseGLObjects(this.commandQueue, glBuffers, computeEvents, null);
            }
        }
    }

    void finish() {
        CL12.clFinish(this.commandQueue);
    }

    private static String getPlatformInfoStringUTF8(long cl_platform_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            PointerBuffer pp = stack.mallocPointer(1);
            OpenCLManager.checkCLError(CL10.clGetPlatformInfo(cl_platform_id, param_name, (ByteBuffer)null, pp));
            int bytes = (int)pp.get(0);
            ByteBuffer buffer = stack.malloc(bytes);
            OpenCLManager.checkCLError(CL10.clGetPlatformInfo(cl_platform_id, param_name, buffer, null));
            String string = MemoryUtil.memUTF8(buffer, bytes - 1);
            return string;
        }
    }

    private static long getDeviceInfoLong(long cl_device_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            LongBuffer pl = stack.mallocLong(1);
            OpenCLManager.checkCLError(CL11.clGetDeviceInfo(cl_device_id, param_name, pl, null));
            long l = pl.get(0);
            return l;
        }
    }

    private static int getDeviceInfoInt(long cl_device_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            IntBuffer pl = stack.mallocInt(1);
            OpenCLManager.checkCLError(CL11.clGetDeviceInfo(cl_device_id, param_name, pl, null));
            int n = pl.get(0);
            return n;
        }
    }

    private static long getDeviceInfoPointer(long cl_device_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            PointerBuffer pp = stack.mallocPointer(1);
            OpenCLManager.checkCLError(CL11.clGetDeviceInfo(cl_device_id, param_name, pp, null));
            long l = pp.get(0);
            return l;
        }
    }

    private static String getDeviceInfoStringUTF8(long cl_device_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            PointerBuffer pp = stack.mallocPointer(1);
            OpenCLManager.checkCLError(CL11.clGetDeviceInfo(cl_device_id, param_name, (ByteBuffer)null, pp));
            int bytes = (int)pp.get(0);
            ByteBuffer buffer = stack.malloc(bytes);
            OpenCLManager.checkCLError(CL11.clGetDeviceInfo(cl_device_id, param_name, buffer, null));
            String string = MemoryUtil.memUTF8(buffer, bytes - 1);
            return string;
        }
    }

    private static int getProgramBuildInfoInt(long cl_program_id, long cl_device_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            IntBuffer pl = stack.mallocInt(1);
            OpenCLManager.checkCLError(CL10.clGetProgramBuildInfo(cl_program_id, cl_device_id, param_name, pl, null));
            int n = pl.get(0);
            return n;
        }
    }

    private static String getProgramBuildInfoStringASCII(long cl_program_id, long cl_device_id, int param_name) {
        try (MemoryStack stack = MemoryStack.stackPush();){
            PointerBuffer pp = stack.mallocPointer(1);
            OpenCLManager.checkCLError(CL10.clGetProgramBuildInfo(cl_program_id, cl_device_id, param_name, (ByteBuffer)null, pp));
            int bytes = (int)pp.get(0);
            ByteBuffer buffer = stack.malloc(bytes);
            OpenCLManager.checkCLError(CL10.clGetProgramBuildInfo(cl_program_id, cl_device_id, param_name, buffer, null));
            String string = MemoryUtil.memASCII(buffer, bytes - 1);
            return string;
        }
    }

    private static void checkCLError(IntBuffer errcode) {
        OpenCLManager.checkCLError(errcode.get(errcode.position()));
    }

    private static void checkCLError(int errcode) {
        if (errcode != 0) {
            throw new RuntimeException(String.format("OpenCL error [%d]", errcode));
        }
    }

    static {
        Configuration.OPENCL_EXPLICIT_INIT.set(true);
    }
}

